using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Globalization;
using System.IO;
using System.ComponentModel; // ListSortDirection
using System.Runtime.InteropServices; // ReversibleSortedList
using System.Diagnostics; // DebuggerDisplay

namespace CSharpRecipes
{
	public class Generics
    {
        #region "4.1 Gdzie i kiedy korzysta z typw generycznych?"
        // Wicej informacji na ten temat mona uzyska w recepturze 4.1 w treci ksiki
		#endregion

        #region "4.2 Podstawowe wiadomoci o typach generycznych"
        public static void TestGenericClassInstanceCounter()
		{
            // Klasa standardowa
            StandardClass A = new StandardClass(1);
            Console.WriteLine(A);
            StandardClass B = new StandardClass(1);
            Console.WriteLine(B);
            StandardClass C = new StandardClass(1);
            Console.WriteLine(C);

            // Klasa generyczna
            GenericClass<bool> gA = new GenericClass<bool>(1);
            Console.WriteLine(gA);
            GenericClass<int> gB = new GenericClass<int>(1);
            Console.WriteLine(gB);
            GenericClass<string> gC = new GenericClass<string>(1);
            Console.WriteLine(gC);
            GenericClass<string> gD = new GenericClass<string>(1);
            Console.WriteLine(gD);
            

            bool b1 = true;
            bool b2 = false;
            bool bHolder = false;

            // dodanie elementw do klasy standardowej (jako object)
            A.AddItem(b1);
            A.AddItem(b2);
            // dodanie elementw do klasy generycznej (jako bool)
            gA.AddItem(b1);
            gA.AddItem(b2);

            Console.WriteLine(A);
            Console.WriteLine(gA);

            // Konieczna konwersja. Jeli nie, bd CS0266: 
            // Cannot implicitly convert type 'object' to 'bool'...
            bHolder = (bool)A.GetItem(1);
            // Konwersja nie jest potrzebna
            bHolder = gA.GetItem(1);

            int i1 = 1;
            int i2 = 2;
            int i3 = 3;
            int iHolder = 0;

            // dodanie elementw do klasy standardowej (jako object)
            B.AddItem(i1);
            B.AddItem(i2);
            B.AddItem(i3);
            // dodanie elementw do klasy generycznej (jako int)
            gB.AddItem(i1);
            gB.AddItem(i2);
            gB.AddItem(i3);

            Console.WriteLine(B);
            Console.WriteLine(gB);

            // Konieczna konwersja. W przypadku jej braku bd CS0266: 
            // Cannot implicitly convert type 'object' to 'int'...
            iHolder = (int)B.GetItem(1);
            // Konwersja nie jest konieczna.
            iHolder = gB.GetItem(1);

            string s1 = "s1";
            string s2 = "s2";
            string s3 = "s3";
            string sHolder = "";

            // dodanie elementw do klasy standardowej (jako object)
            C.AddItem(s1);
            C.AddItem(s2);
            C.AddItem(s3);
            // dodanie elementu int do egzemplarza string - cakowicie poprawne
            C.AddItem(i1);

            // dodanie do klasy generycznej (jako cig znakw)
            gC.AddItem(s1);
            gC.AddItem(s2);
            gC.AddItem(s3);
            // prba dodania do egzemplarza string zabroniona przez kompilator
            // error CS1503: Argument '1': cannot convert from 'int' to 'string'
            //gC.AddItem(i1);

            Console.WriteLine(C);
            Console.WriteLine(gC);

            // Konieczna konwersja. Przy jej braku bd CS0266: 
            // Cannot implicitly convert type 'object' to 'string'...
            sHolder = (string)C.GetItem(1);
            // Konwersja nie jest potrzebna
            sHolder = gC.GetItem(1);
            // bd: prba przypisania cigu znakw do danej int
            // error CS0029: Cannot implicitly convert type 'string' to 'int'
            //iHolder = gC.GetItem(1);
        }

        public class StandardClass
        {
            // Statyczny licznik egzemplarzy klasy
            // StandardClass.
            static int _count = 0;

            // Utworzenie tablicy elementw typu.
            int _maxItemCount;
            object[] _items;
            int _currentItem = 0;

            // Konstruktor inkrementujcy statyczny licznik
            public StandardClass(int items)
            {
                _count++;
                _maxItemCount = items;
                _items = new object[_maxItemCount];
            }

            /// <summary>
            /// Dodaje element klasy, ktrej typ
            /// jest nieznany, poniewa tylko klasa Object pozwala na zapisywanie dowolnych typw.
            /// </summary>
            /// <param name="item">pozycja do dodania</param>
            /// <returns>indeks dodanej pozycji</returns>
            public int AddItem(object item)
            {
                if (_currentItem < _maxItemCount)
                {
                    _items[_currentItem] = item;
                    return _currentItem++;
                }
                else
                    throw new Exception("Kolejka elementw jest pena");
            }

            /// <summary>
            /// Pobranie elementu klasy.
            /// </summary>
            /// <param name="index">indeks elementu do pobrania</param>
            /// <returns>element typu object</returns>
            public object GetItem(int index)
            {
                Debug.Assert(index < _maxItemCount);
                if (index >= _maxItemCount)
                    throw new ArgumentOutOfRangeException("index");

                return _items[index];
            }

            /// <summary>
            /// Licznik elementw klasy
            /// </summary>
            public int ItemCount
            {
                get { return _currentItem; }
            }

            /// <summary>
            /// Przeciona wersja metody ToString wywietlajca informacje na temat klasy
            /// </summary>
            /// <returns>sformatowany cig znakw zawierajcy informacje na temat klasy</returns>
            public override string ToString()
            {
                return "Klasa obejmuje " + _count.ToString() +
                " egzemplarzy " + this.GetType().ToString() +
                " ktre zawieraj " + _currentItem + " elementw typu " +
                _items.GetType().ToString() + "...";
            }
        }

        public class GenericClass<T>
        {
            // Statyczny licznik egzemplarzy klasy
            // GenericClass.
            static int _count = 0;

            // Utworzenie tablicy elementw typu.
            int _maxItemCount;
            T[] _items;
            int _currentItem = 0;

            // Konstruktor inkrementujcy statyczny licznik
            public GenericClass(int items)
            {
                _count++;
                _maxItemCount = items;
                _items = new T[_maxItemCount];
            }

            /// <summary>
            /// Dodaje element klasy, ktrej typ
            /// jest okrelony typem egzemplarza.
            /// </summary>
            /// <param name="item">element do dodania</param>
            /// <returns>indeks dodanej pozycji (poczwszy od zera)</returns>
            public int AddItem(T item)
            {
                if (_currentItem < _maxItemCount)
                {
                    _items[_currentItem] = item;
                    return _currentItem++;
                }
                else
                    throw new Exception("Kolejka elementw jest pena");
            }

            /// <summary>
            /// Pobranie egzemplarza klasy.
            /// </summary>
            /// <param name="index">indeks elementu do pobrania (poczwszy od zera)</param>
            /// <returns>egzemplarz typu</returns>
            public T GetItem(int index)
            {
                Debug.Assert(index < _maxItemCount);
                if (index >= _maxItemCount)
                    throw new ArgumentOutOfRangeException("index");

                return _items[index];
            }

            /// <summary>
            /// Licznik egzemplarzy klasy
            /// </summary>
            public int ItemCount
            {
                get { return _currentItem; }
            }

            /// <summary>
            /// Przeciona wersja metody ToString wywietlajca informacje na temat klasy
            /// </summary>
            /// <returns>sformatowany cig znakw zawierajcy informacje na temat klasy</returns>
            public override string ToString()
            {
                return "Klasa obejmuje " + _count.ToString() +
                    " egzemplarzy " + this.GetType().ToString() +
                    " ktre zawieraj " + _currentItem + " elementw typu " +
                     _items.GetType().ToString() + "...";
            }
        }

		#endregion

        #region "4.3 Odczytywanie obiektu Type dla danych typu generycznego
        public static void TestGetGenTypeOf()
		{
			Simple s = new Simple();

			Type t = typeof(Simple);
			Type alsoT = s.GetType();

			// Nie mona uzyska typu klasy generycznej,
			// poniewa klasy generyczne nie maj typu
			// jeli nie poda si parametru.  Typ mona wyznaczy 
			// tylko dla egzemplarzy klas generycznych z podanym
			// parametrem opisujcym typ.

			// Wykonanie instrukcji spowoduje bd:
			//Error	26	Using the generic type 'CSharpRecipes.Generics.SimpleGeneric<T>' requires '1' type arguments
			//Type gt = typeof(SimpleGeneric);

			// Po podaniu parametru opisujcego typ, 
			// mona uzyska typ egzemplarza
			Type gtInt = typeof(SimpleGeneric<int>);
			Type gtBool = typeof(SimpleGeneric<bool>);
			Type gtString = typeof(SimpleGeneric<string>);

			// Mona rwnie posuy si standardowym wywoaniem metody GetType egzemplarza,
            // poniewa obiekt musi by egzemplarzem typu generycznego.

			SimpleGeneric<int> sgI = new SimpleGeneric<int>();
			Type alsoGT = sgI.GetType();
		}

		public class Simple
		{
			public Simple()
			{
			}
		}

		public class SimpleGeneric<T>
		{
			public SimpleGeneric()
			{
			}
		}
		#endregion

        #region "4.4 Zastpowanie typu ArrayList jego generycznym odpowiednikiem"
        public static void UseNonGenericArrayList()
		{
			Console.WriteLine("\r\nUseNonGenericList");

            // Utworzenie obiektu ArrayList i wypenienie go danymi.
			ArrayList numbers = new ArrayList();
            numbers.Add(1);    // Powoduje wykonanie operacji pakowania.
            numbers.Add(2);    // Powoduje wykonanie operacji pakowania.

            // Wywietlenie wszystkich danych typu integer w ArrayList.
            // powoduje wykonywanie operacji rozpakowania w kadej iteracji

			foreach (int i in numbers)
			{
				Console.WriteLine(i);
			}

			numbers.Clear();

			Console.WriteLine(numbers.IsReadOnly);
			Console.WriteLine(numbers.IsFixedSize);
			Console.WriteLine(numbers.IsSynchronized);
			Console.WriteLine(numbers.SyncRoot);
		}

		public static void UseGenericList()
		{
			Console.WriteLine("\r\nUseGenericList");

            // Utworzenie listy i wypenienie jej danymi.
			List<int> numbers = new List<int>();
			numbers.Add(1);
			numbers.Add(2);

            // Wywietlenie wszystkich danych typu integer w ArrayList.
			foreach (int i in numbers)
			{
				Console.WriteLine(i);
			}

			numbers.Clear();

			Console.WriteLine(((IList<int>)numbers).IsReadOnly);
			Console.WriteLine(((IList)numbers).IsFixedSize);
			Console.WriteLine(((IList)numbers).IsSynchronized);
			Console.WriteLine(((IList)numbers).SyncRoot);
		}

		public static void TestAdapterVsCtor()
		{
			Console.WriteLine("\r\nTest metody Adapter");

			ArrayList al = new ArrayList();
			al.Add(new object());
			ArrayList adapter = ArrayList.Adapter(al);
			Console.WriteLine("al " + al[0].GetHashCode());
			Console.WriteLine("adapter " + adapter[0].GetHashCode());

			List<object> oldList = new List<object>();
			oldList.Add(new object());
			List<object> newList = new List<object>(oldList);
			Console.WriteLine("oldList " + oldList[0].GetHashCode());
			Console.WriteLine("newList " + newList[0].GetHashCode());
		}

		public static void TestCloneVsGetRange()
		{
			Console.WriteLine("\r\nTest metody Clone");

			ArrayList al = new ArrayList();
			al.Add(new object());
			ArrayList clone = (ArrayList)al.Clone();
			Console.WriteLine("al " + al[0].GetHashCode());
			Console.WriteLine("clone " + clone[0].GetHashCode());

			List<object> oldList = new List<object>();
			oldList.Add(new object());
			List<object> newList = oldList.GetRange(0, oldList.Count);
			Console.WriteLine("oldList " + oldList[0].GetHashCode());
			Console.WriteLine("newList " + newList[0].GetHashCode());
		}

		public static void TestGenericRepeat()
		{
			List<int> numberList = new List<int>();
			Repeat(numberList, 100, 3);

			foreach (int i in numberList)
				Console.WriteLine(i);
		}

		public static void Repeat<T>(List<T> list, T obj, int count)
		{
			if (count < 0)
			{
				throw (new ArgumentException("Parametr count parameter musi by wikszy bd rwny zero."));
			}

			for (int index = 0; index < count; index++)
			{
				list.Add(obj);
			}
		}

		public static void CloneGenericList()
		{
			Console.WriteLine("\r\nCloneGenericList");

			List<int> numbers = new List<int>();
			numbers.Add(1);
			numbers.Add(2);

			List<int> cloneNumbers = new List<int>(numbers);
			Console.WriteLine("cloneNumbers[0] " + cloneNumbers[0].GetHashCode());

			Console.WriteLine(((IList<int>)numbers).IsReadOnly);
			Console.WriteLine(((IList)numbers).IsFixedSize);
			Console.WriteLine(((IList)numbers).IsSynchronized);
			Console.WriteLine(((IList)numbers).SyncRoot);
		}
		#endregion

        #region "4.5 Zastpienie obiektw Stack i Queue ich generycznymi odpowiednikami"
        public static void UseGenericStack()
		{
            //  Utworzenie generycznego obiektu Stack.
			Stack<int> numericStack = new Stack<int>();

            // Wypenienie obiektu Stack danymi
			numericStack.Push(1);
			numericStack.Push(2);

			double s = 3;
			numericStack.Push((int)s);

            // To jest jedynie wywietlanie elementw stosu, a nie ich zdejmowanie
			foreach (int i in numericStack)
			{
				Console.WriteLine("foreach: " + i.ToString());
			}

            // Pobieranie danych z obiektu Stack i wywietlanie ich.
			Console.WriteLine(numericStack.Pop().ToString());
			Console.WriteLine(numericStack.Pop().ToString());
			Console.WriteLine(numericStack.Pop().ToString());
		}

		public static void UseNonGenericStack()
		{
            //  Utworzenie niegenerycznego obiektu Stack.
			Stack numericStack = new Stack();

            //  Wypenienie obiektu Stack danymi (powoduje operacj pakowania).
			numericStack.Push(1);
			numericStack.Push(2);
			numericStack.Push(3);

			// Podgldanie elementw odoonych na stosie bez ich zdejmowania
			foreach (int i in numericStack)
			{
				Console.WriteLine("foreach: " + i.ToString());
			}

            //  Pobieranie danych z obiektu Stack i ich wywietlanie (powoduje wykonywanie operacji rozpakowania)
			Console.WriteLine(numericStack.Pop().ToString());
			Console.WriteLine(numericStack.Pop().ToString());
			Console.WriteLine(numericStack.Pop().ToString());
		}

		public static void CloneStack()
		{
            //  Utworzenie generycznego obiektu Stack.
			Stack<int> numericStack = new Stack<int>();

            //  Wypenienie obiektu Stack danymi
			numericStack.Push(1);
			numericStack.Push(2);
			numericStack.Push(3);

			// Sklonowanie obiektu numericStack
			Stack<int> clonedNumericStack = new Stack<int>(numericStack);

			// Podgldanie elementw stosu bez ich zdejmowania
			foreach (int i in clonedNumericStack)
			{
				Console.WriteLine("foreach: " + i.ToString());
			}

            // Pobieranie danych z obiektu Stack i wywietlanie ich.
			Console.WriteLine(clonedNumericStack.Pop().ToString());
			Console.WriteLine(clonedNumericStack.Pop().ToString());
			Console.WriteLine(clonedNumericStack.Pop().ToString());
		}


		public static void UseGenericQueue()
		{
            //  Utworzenie generycznego obiektu Queue.
			Queue<int> numericQueue = new Queue<int>();

            //  Wypenienie obiektu Queue danymi.
			numericQueue.Enqueue(1);
			numericQueue.Enqueue(2);

			double s = 3;
			numericQueue.Enqueue((int)s);

			// Podgldanie elementw kolejki bez ich usuwania
			foreach (int i in numericQueue)
			{
				Console.WriteLine("foreach: " + i.ToString());
			}

            // Pobieranie danych z obiektu Queue i wywietlanie ich.
			Console.WriteLine(numericQueue.Dequeue().ToString());
			Console.WriteLine(numericQueue.Dequeue().ToString());
			Console.WriteLine(numericQueue.Dequeue().ToString());
		}

		public static void UseNonGenericQueue()
		{
            // Utworzenie niegenerycznego obiektu Queue.
			Queue numericQueue = new Queue();

            // Wypenienie obiektu Queue danymi (powoduje operacj pakowania).
			numericQueue.Enqueue(1);
			numericQueue.Enqueue(2);
			numericQueue.Enqueue(3);

			// Podgldanie elementw w kolejce bez ich usuwania.
			foreach (int i in numericQueue)
			{
				Console.WriteLine("foreach: " + i.ToString());
			}

            //  Pobieranie danych z obiektu Queue i ich wywietlanie (powoduje wykonywanie operacji rozpakowania).
			Console.WriteLine(numericQueue.Dequeue().ToString());
			Console.WriteLine(numericQueue.Dequeue().ToString());
			Console.WriteLine(numericQueue.Dequeue().ToString());
		}

		public static void CloneQueue()
		{
            // Utworzenie generycznego obiektu Queue.
			Queue<int> numericQueue = new Queue<int>();

			// Wypenienie obiektu Queue danymi
			numericQueue.Enqueue(1);
			numericQueue.Enqueue(2);
			numericQueue.Enqueue(3);

			// Sklonowanie obiektu numericQueue
			Queue<int> clonedNumericQueue = new Queue<int>(numericQueue);

			// Podgld wartoci w kolejce bez ich usuwania
			foreach (int i in clonedNumericQueue)
			{
				Console.WriteLine("foreach: " + i.ToString());
			}

            //  Pobieranie danych z obiektu Queue i ich wywietlanie.
			Console.WriteLine(clonedNumericQueue.Dequeue().ToString());
			Console.WriteLine(clonedNumericQueue.Dequeue().ToString());
			Console.WriteLine(clonedNumericQueue.Dequeue().ToString());
		}
		#endregion

        #region "4.6 Implementacja powizanych list"
        public static void UseLinkedList()
		{
			Console.WriteLine("\r\n\r\n");

            //  Utworzenie obiektu LinkedList.
			LinkedList<TodoItem> todoList = new LinkedList<TodoItem>();

            //  Utworzenie obiektw TodoItem w celu dodania ich do powizanej listy.
            TodoItem i1 = new TodoItem("malowanie drzwi", "Naley wykona w trzeciej kolejnoci");
            TodoItem i2 = new TodoItem("zakup drzwi", "Naley wykona w pierwszej kolejnoci");
            TodoItem i3 = new TodoItem("przygotowanie drzwi do montau", "Naley wykona w drugiej kolejnoci");
            TodoItem i4 = new TodoItem("monta drzwi", "Naley wykona w ostatniej kolejnoci");

            // Dodanie elementw.

			todoList.AddFirst(i1);
			todoList.AddFirst(i2);
			todoList.AddBefore(todoList.Find(i1), i3);
			todoList.AddAfter(todoList.Find(i1), i4);

            // Wywietlenie wszystkich elementw.
            foreach (TodoItem tdi in todoList)
			{
				Console.WriteLine(tdi.Name + " : " + tdi.Comment);
			}

            // Wywietlenie informacji z pierwszego wza powizanej listy.
			Console.WriteLine("todoList.First.Value.Name == " + 
todoList.First.Value.Name);

            // Wywietlenie informacji z drugiego wza powizanej listy.
			Console.WriteLine("todoList.First.Next.Value.Name == " + 
todoList.First.Next.Value.Name);

            // Wywietlenie informacji od nastpnego do ostatniego wza powizanej listy.
			Console.WriteLine("todoList.Last.Previous.Value.Name == " + 
todoList.Last.Previous.Value.Name);
		}

		public class TodoItem
		{
			public TodoItem (string name, string comment)
			{
				_name = name;
				_comment = comment;
			}

			private string _name = "";
			private string _comment = "";

			public string Name
			{
				get	{return (_name);}
				set {_name = value;}
			}

			public string Comment
			{
				get	{return (_comment);}
				set	{_comment = value;}
			}
		}
		#endregion

        #region "4.7  Tworzenie typu wartoci, ktry mona zainicjowa wartoci null"
        public static void TestNullableStruct()
		{
			Console.WriteLine("\r\n\r\n");

			//int? myDBInt = null;
			//  OR
			Nullable<int> myDBInt = new Nullable<int>();
			Nullable<int> myTempDBInt = new Nullable<int>();


            if (myDBInt.HasValue)
                Console.WriteLine("Ma warto:    " + myDBInt.Value);
            else
                Console.WriteLine("Nie ma wartoci (NULL)");

			myDBInt = 100;

            if (myDBInt != null)
                Console.WriteLine("Ma warto:    " + myDBInt.Value);
            else
                Console.WriteLine("Nie ma wartoci (NULL)");

			if (myTempDBInt != null)
			{
				if (myTempDBInt < 100)
					Console.WriteLine("myTempDBInt < 100");
				else
					Console.WriteLine("myTempDBInt >= 100");
			}
			else
			{
                  // Obsuga wartoci null.	
            }
		}
		#endregion

		#region "4.8 Odwrcenie porzdku posortowanej listy"
[Serializable, ComVisible(false), DebuggerDisplay("Count = {Count}")]
        public class ReversibleSortedList<TKey, TValue> : IDictionary<TKey, TValue>, ICollection<KeyValuePair<TKey, TValue>>, IEnumerable<KeyValuePair<TKey, TValue>>, IDictionary, ICollection, IEnumerable
{
    #region SortDirectionComparer class definition
    public class SortDirectionComparer<T> : IComparer<T>
    {
        private System.ComponentModel.ListSortDirection _sortDir;

        public SortDirectionComparer()
        {
             _sortDir = ListSortDirection.Ascending;
        }

        public SortDirectionComparer(ListSortDirection sortDir)
        {
            _sortDir = sortDir;
        }

        public System.ComponentModel.ListSortDirection SortDirection
        {
             get { return _sortDir; }
             set { _sortDir = value; }
        }

        public int Compare(T lhs, T rhs)
        {
             int compareResult =
                 lhs.ToString().CompareTo(rhs.ToString());

             // W przypadku porzdku malejcego naley odwrci to porwnanie
             if (SortDirection == ListSortDirection.Descending)
                compareResult *= -1;
             return compareResult;
        }
    }
    #endregion // SortDirectionComparer

    #region CTORS
    static ReversibleSortedList()
    {
        ReversibleSortedList<TKey, TValue>.emptyKeys = new TKey[0];
        ReversibleSortedList<TKey, TValue>.emptyValues = new TValue[0];
    }

    public ReversibleSortedList()
    {
        this.keys = ReversibleSortedList<TKey, TValue>.emptyKeys;
        this.values = ReversibleSortedList<TKey, TValue>.emptyValues;
        this._size = 0;
        this._sortDirectionComparer = new SortDirectionComparer<TKey>();
        this._currentSortDirection = this._sortDirectionComparer.SortDirection;
    }

    public ReversibleSortedList(SortDirectionComparer<TKey> comparer)
          : this()
    {
        if (comparer != null)
        {
            this._sortDirectionComparer = comparer;
            this._currentSortDirection = _sortDirectionComparer.SortDirection;
        }
    }

    public ReversibleSortedList(IDictionary<TKey, TValue> dictionary)
       : this(dictionary, (SortDirectionComparer<TKey>)null)
    {
    }

    public ReversibleSortedList(int capacity)
    {
        if (capacity < 0)
        {
            throw new ArgumentOutOfRangeException("capacity", "Wymagana liczba nieujemna");
        }
        this.keys = new TKey[capacity];
        this.values = new TValue[capacity];
        this._sortDirectionComparer = new SortDirectionComparer<TKey>();
        this._currentSortDirection = _sortDirectionComparer.SortDirection;
    }

    public ReversibleSortedList(IDictionary<TKey, TValue> dictionary, SortDirectionComparer<TKey> comparer)
          : this((dictionary != null) ? dictionary.Count : 0, comparer)
    {
        if (dictionary == null)
        {
             throw new ArgumentNullException("dictionary");
        }
        dictionary.Keys.CopyTo(this.keys, 0);
        dictionary.Values.CopyTo(this.values, 0);
        Array.Sort<TKey, TValue>(this.keys, this.values, this._sortDirectionComparer);
        this._size = dictionary.Count;
    }

    public ReversibleSortedList(int capacity, SortDirectionComparer<TKey> comparer)
          : this(comparer)
    {
        this.Capacity = capacity;
    }
    #endregion //CTORS

    #region Metody publiczne
    public void Add(TKey key, TValue value)
    {
       if (key.Equals(null))
       {
           throw new ArgumentNullException("key");
       }
       int num1 = Array.BinarySearch<TKey>(this.keys, 0, this._size, key, this._sortDirectionComparer);
       if (num1 >= 0)
       {
             throw new ArgumentException("Prba dodania duplikatu");
        }
        this.Insert(~num1, key, value);
    }

    public void Clear()
    {
        this.version++;
        Array.Clear(this.keys, 0, this._size);
        Array.Clear(this.values, 0, this._size);
        this._size = 0;
    }

    public bool ContainsKey(TKey key)
    {
        return (this.IndexOfKey(key) >= 0);
    }

    public bool ContainsValue(TValue value)
    {
        return (this.IndexOfValue(value) >= 0);
    }

    public IEnumerator<KeyValuePair<TKey, TValue>> GetEnumerator()
    {
        return new ReversibleSortedList<TKey, TValue>.Enumerator<TKey, TValue>(this);
    }
    public int IndexOfKey(TKey key)
    {
        if (key.Equals(null))
        {
            throw new ArgumentNullException("key");
        }
        int num1 = Array.BinarySearch<TKey>(this.keys, 0, this._size, key, this._sortDirectionComparer);
        if (num1 < 0)
        {
            return -1;
        }
        return num1;
    }

    public int IndexOfValue(TValue value)
    {
        return Array.IndexOf<TValue>(this.values, value, 0, this._size);
    }

    public bool Remove(TKey key)
    {
        int num1 = this.IndexOfKey(key);
        if (num1 >= 0)
        {
            this.RemoveAt(num1);
        }
        return (num1 >= 0);
    }

    public void RemoveAt(int index)
    {
        if ((index < 0) || (index >= this._size))
        {
             throw new ArgumentOutOfRangeException("index", "Indeks poza zakresem");
        }
        this._size--;
        if (index < this._size)
        {
            Array.Copy(this.keys, (int)(index + 1), this.keys, index, (int)(this._size - index));
            Array.Copy(this.values, (int)(index + 1), this.values, index, (int)(this._size - index));
        }
        this.keys[this._size] = default(TKey);
        this.values[this._size] = default(TValue);
        this.version++;
    }

    public void Sort()
    {
        // Sprawdzenie, czy biecy porzdek sortowania jest waciwy
        if (this._currentSortDirection != this._sortDirectionComparer.SortDirection)
        {
            // Odwrcenie tablic - byy posortowane ju w momencie dodawanie elemntu
            Array.Reverse(this.keys, 0, this._size);
            Array.Reverse(this.values, 0, this._size);
            // Ustawienie biecego porzdku
            this._currentSortDirection = this._sortDirectionComparer.SortDirection;
        }
    }

    public void TrimExcess()
    {
        int num1 = (int)(this.keys.Length * 0.9);
        if (this._size < num1)
        {
            this.Capacity = this._size;
        }
    }

    public bool TryGetValue(TKey key, out TValue value)
    {
        int num1 = this.IndexOfKey(key);
        if (num1 >= 0)
        {
            value = this.values[num1];
            return true;
        }
        value = default(TValue);
        return false;
    }

    #endregion // Metody publiczne
    #region Metody prywatne
    private void EnsureCapacity(int min)
    {
        int num1 = (this.keys.Length == 0) ? 4 : (this.keys.Length * 2);
        if (num1 < min)
        {
            num1 = min;
        }
        this.InternalSetCapacity(num1, false);
    }

    private TValue GetByIndex(int index)
    {
        if ((index < 0) || (index >= this._size))
        {
            throw new ArgumentOutOfRangeException("index", "Indeks poza zakresem");
        }
        return this.values[index];
    }

    private TKey GetKey(int index)
    {
        if ((index < 0) || (index >= this._size))
        {
            throw new ArgumentOutOfRangeException("index", "Indeks poza zakresem");
        }
        return this.keys[index];
    }

    private KeyList<TKey, TValue> GetKeyListHelper()
    {
        if (this.keyList == null)
        {
            this.keyList = new KeyList<TKey, TValue>(this);
        }
        return this.keyList;
    }

    private ValueList<TKey, TValue> GetValueListHelper()
    {
        if (this.valueList == null)
        {
            this.valueList = new ValueList<TKey, TValue>(this);
        }
        return this.valueList;
    }

    private void Insert(int index, TKey key, TValue value)
    {
        if (this._size == this.keys.Length)
        {
            this.EnsureCapacity(this._size + 1);
        }
        if (index < this._size)
        {
            Array.Copy(this.keys, index, this.keys, (int)(index + 1), (int)(this._size - index));
            Array.Copy(this.values, index, this.values, (int)(index + 1), (int)(this._size - index));
        }
        this.keys[index] = key;
        this.values[index] = value;
        this._size++;
        this.version++;
    }

    private void InternalSetCapacity(int value, bool updateVersion)
    {
        if (value != this.keys.Length)
        {
            if (value < this._size)
            {
                throw new ArgumentOutOfRangeException("value", "Zbyt maa objto");
            }
            if (value > 0)
            {
                TKey[] localArray1 = new TKey[value];
                TValue[] localArray2 = new TValue[value];
                if (this._size > 0)
                {
                    Array.Copy(this.keys, 0, localArray1, 0, this._size);
                    Array.Copy(this.values, 0, localArray2, 0, this._size);
                }
                this.keys = localArray1;
                this.values = localArray2;
            }
            else
            {
                this.keys = ReversibleSortedList<TKey, TValue>.emptyKeys;
                this.values = ReversibleSortedList<TKey, TValue>.emptyValues;
            }
            if (updateVersion)
            {
                this.version++;
            }
        }
    }

    private static bool IsCompatibleKey(object key)
    {
        if (key == null)
        {
            throw new ArgumentNullException("key");
        }
        return (key is TKey);
    }

    void ICollection<KeyValuePair<TKey, TValue>>.Add(KeyValuePair<TKey, TValue> keyValuePair)
    {
        this.Add(keyValuePair.Key, keyValuePair.Value);
    }

    bool ICollection<KeyValuePair<TKey, TValue>>.Contains(KeyValuePair<TKey, TValue> keyValuePair)
    {
        int num1 = this.IndexOfKey(keyValuePair.Key);
        if ((num1 >= 0) && EqualityComparer<TValue>.Default.Equals(this.values[num1], keyValuePair.Value))
        {
            return true;
        }
        return false;
    }

    void ICollection<KeyValuePair<TKey, TValue>>.CopyTo(KeyValuePair<TKey, TValue>[] array, int arrayIndex)
    {
        if (array == null)
        {
            throw new ArgumentNullException("array");
        }
        if ((arrayIndex < 0) || (arrayIndex > array.Length))
        {
            throw new ArgumentOutOfRangeException("arrayIndex", "Indeks musi by liczb nieujemn");
        }
        if ((array.Length - arrayIndex) < this.Count)
        {
            throw new ArgumentException("ArrayPlusOffTooSmall");
        }
        for (int num1 = 0; num1 < this.Count; num1++)
        {
            KeyValuePair<TKey, TValue> pair1;
            pair1 = new KeyValuePair<TKey, TValue>(this.keys[num1], this.values[num1]);
            array[arrayIndex + num1] = pair1;
        }
    }

    bool ICollection<KeyValuePair<TKey, TValue>>.Remove(KeyValuePair<TKey, TValue> keyValuePair)
    {
        int num1 = this.IndexOfKey(keyValuePair.Key);
        if ((num1 >= 0) && EqualityComparer<TValue>.Default.Equals(this.values[num1], keyValuePair.Value))
        {
            this.RemoveAt(num1);
            return true;
        }
        return false;
    }

    IEnumerator<KeyValuePair<TKey, TValue>> IEnumerable<KeyValuePair<TKey, TValue>>.GetEnumerator()
    {
        return new ReversibleSortedList<TKey, TValue>.Enumerator<TKey, TValue>(this);
    }

    void ICollection.CopyTo(Array array, int arrayIndex)
    {
        if (array == null)
        {
            throw new ArgumentNullException("array");
        }
        if (array.Rank != 1)
        {
            throw new ArgumentException("Kopie tablic wielowymiarowych nie s obsugiwane");
        }
        if (array.GetLowerBound(0) != 0)
        {
            throw new ArgumentException("Wprowadzono dolny indeks rny od zera");
        }
        if ((arrayIndex < 0) || (arrayIndex > array.Length))
        {
            throw new ArgumentOutOfRangeException("arrayIndex", "Indeks tablicy musi by liczb nieujemn");
        }
        if ((array.Length - arrayIndex) < this.Count)
        {
            throw new ArgumentException("Warto sumy rozmiaru tablicy i przesunicia jest zbyt maa");
        }
        KeyValuePair<TKey, TValue>[] pairArray1 = array as KeyValuePair<TKey, TValue>[];
        if (pairArray1 != null)
        {
            for (int num1 = 0; num1 < this.Count; num1++)
            {
                pairArray1[num1 + arrayIndex] = new KeyValuePair<TKey, TValue>(this.keys[num1], this.values[num1]);
            }
        }
        else
        {
            object[] objArray1 = array as object[];
            if (objArray1 == null)
            {
                throw new ArgumentException("Nieprawidowy typ tablicy");
            }
            try
            {
                for (int num2 = 0; num2 < this.Count; num2++)
                {
                    objArray1[num2 + arrayIndex] = new KeyValuePair<TKey, TValue>(this.keys[num2], this.values[num2]);
                }
            }
            catch (ArrayTypeMismatchException)
            {
                throw new ArgumentException("Nieprawidowy typ tablicy");
            }
        }
    }

    void IDictionary.Add(object key, object value)
    {
        ReversibleSortedList<TKey, TValue>.VerifyKey(key);
        ReversibleSortedList<TKey, TValue>.VerifyValueType(value);
        this.Add((TKey)key, (TValue)value);
    }

    bool IDictionary.Contains(object key)
    {
        if (ReversibleSortedList<TKey, TValue>.IsCompatibleKey(key))
        {
            return this.ContainsKey((TKey)key);
        }
        return false;
    }

    IDictionaryEnumerator IDictionary.GetEnumerator()
    {
        return new ReversibleSortedList<TKey, TValue>.Enumerator<TKey, TValue>(this);
    }

    void IDictionary.Remove(object key)
    {
        if (ReversibleSortedList<TKey, TValue>.IsCompatibleKey(key))
        {
            this.Remove((TKey)key);
        }
    }

    IEnumerator IEnumerable.GetEnumerator()
    {
        return new ReversibleSortedList<TKey, TValue>.Enumerator<TKey, TValue>(this);
    }

    private static void VerifyKey(object key)
    {
        if (key == null)
        {
            throw new ArgumentNullException("key");
        }
        if (!(key is TKey))
        {
            throw new ArgumentException("Przekazany argument ma nieprawidowy typ", "key");
        }
    }

    private static void VerifyValueType(object value)
    {
        if (!(value is TValue) && ((value != null) || typeof(TValue).IsValueType))
        {
            throw new ArgumentException("Przekazany argument jest nieprawidowego typu", "value");
        }
    }
    #endregion // Metody prywatne

    #region Waciwoci publiczne
    public int Capacity
    {
        get
        {
            return this.keys.Length;
        }
        set
        {
            this.InternalSetCapacity(value, true);
        }
    }

    public SortDirectionComparer<TKey> Comparer
    {
        get
        {
            return this._sortDirectionComparer;
        }
    }

    public int Count
    {
        get
        {
            return this._size;
        }
    }

    public TValue this[TKey key]
    {
        get
        {
            TValue local1;
            int num1 = this.IndexOfKey(key);
            if (num1 >= 0)
            {
                return this.values[num1];
            }
            else
            {
                //zgoszenie wyjtku KeyNotFoundException();
                local1 = default(TValue);
                return local1;
            }
        }
        set
        {
            if (key.Equals(null))
            {
                throw new ArgumentNullException("key");
            }
            int num1 = Array.BinarySearch<TKey>(this.keys, 0, this._size, key, this._sortDirectionComparer);
            if (num1 >= 0)
            {
                this.values[num1] = value;
                this.version++;
            }
            else
            {
                this.Insert(~num1, key, value);
            }
        }
    }

    public IList<TKey> Keys
    {
        get
        {
            return this.GetKeyListHelper();
        }
    }

    public IList<TValue> Values
    {
        get
        {
            return this.GetValueListHelper();
        }
    }
    #endregion // Waciwoci publiczne

    #region Waciwoci prywatne
    bool ICollection<KeyValuePair<TKey, TValue>>.IsReadOnly
    {
        get
        {
            return false;
        }
    }

    ICollection<TKey> IDictionary<TKey, TValue>.Keys
    {
        get
        {
            return this.GetKeyListHelper();
        }
    }

    ICollection<TValue> IDictionary<TKey, TValue>.Values
    {
        get
        {
            return this.GetValueListHelper();
        }
    }

    bool ICollection.IsSynchronized
    {
        get
        {
            return false;
        }
    }

    object ICollection.SyncRoot
    {
        get
        {
            return this;
        }
    }

    bool IDictionary.IsFixedSize
    {
        get
        {
            return false;
        }
    }

    bool IDictionary.IsReadOnly
    {
        get
        {
            return false;
        }
    }

    object IDictionary.this[object key]
    {
        get
        {
            if (ReversibleSortedList<TKey, TValue>.IsCompatibleKey(key))
            {
                int num1 = this.IndexOfKey((TKey)key);
                if (num1 >= 0)
                {
                    return this.values[num1];
                }
            }
            return null;
        }
        set
        {
            ReversibleSortedList<TKey, TValue>.VerifyKey(key);
            ReversibleSortedList<TKey, TValue>.VerifyValueType(value);
            this[(TKey)key] = (TValue)value;
        }
    }

    ICollection IDictionary.Keys
    {
        get
        {
            return this.GetKeyListHelper();
        }
    }

    ICollection IDictionary.Values
    {
        get
        {
            return this.GetValueListHelper();
        }
    }
    #endregion // Waciwoci prywatne
    #region Pola
    private const int _defaultCapacity = 4;
    private int _size;
    //private IComparer<TKey> comparer;
    private static TKey[] emptyKeys;
    private static TValue[] emptyValues;
    private KeyList<TKey, TValue> keyList;
    private TKey[] keys;
    private ValueList<TKey, TValue> valueList;
    private TValue[] values;
    private int version;
    // deklaracja obiektu porwnujcego
    private SortDirectionComparer<TKey> _sortDirectionComparer = null;
    // domylnie porzdek rosncy
    private ListSortDirection _currentSortDirection = ListSortDirection.Descending;
    #endregion
    #region Typy zagniedone

    #region Enumerator <K, V>
    [Serializable, StructLayout(LayoutKind.Sequential)]
    private struct Enumerator<K, V> : IEnumerator<KeyValuePair<K, V>>, IDisposable, IDictionaryEnumerator, IEnumerator
    {
        private ReversibleSortedList<K, V> _ReversibleSortedList;
        private K key;
        private V value;
        private int index;
        private int version;
        internal Enumerator(ReversibleSortedList<K, V> ReversibleSortedList)
        {
            this._ReversibleSortedList = ReversibleSortedList;
            this.index = 0;
            this.version = this._ReversibleSortedList.version;
            this.key = default(K);
            this.value = default(V);
        }
        public void Dispose()
        {
            this.index = 0;
            this.key = default(K);
            this.value = default(V);
        }
        object IDictionaryEnumerator.Key
        {
            get
            {
                if ((this.index == 0) || (this.index == (this._ReversibleSortedList.Count + 1)))
                {
                    throw new InvalidOperationException("Nie mona wykona operacji wyliczania.");
                }
                return this.key;
            }
        }
        public bool MoveNext()
        {
            if (this.version != this._ReversibleSortedList.version)
            {
                throw new InvalidOperationException("Bd kontroli wersji typu wyliczeniowego");
            }
            if (this.index < this._ReversibleSortedList.Count)
            {
                this.key = this._ReversibleSortedList.keys[this.index];
                this.value = this._ReversibleSortedList.values[this.index];
                this.index++;
                return true;
            }
            this.index = this._ReversibleSortedList.Count + 1;
            this.key = default(K);
            this.value = default(V);
            return false;
        }
        DictionaryEntry IDictionaryEnumerator.Entry
        {
            get
            {
                if ((this.index == 0) || (this.index == (this._ReversibleSortedList.Count + 1)))
                {
                    throw new InvalidOperationException("Niedozwolona operacja wyliczania.");
                }
                return new DictionaryEntry(this.key, this.value);
            }
        }
        public KeyValuePair<K, V> Current
        {
            get
            {
                return new KeyValuePair<K, V>(this.key, this.value);
            }
        }
        object IEnumerator.Current
        {
            get
            {
                if ((this.index == 0) || (this.index == (this._ReversibleSortedList.Count + 1)))
                {
                    throw new InvalidOperationException("Nie mona wykona operacji wyliczania.");
                }
                return new DictionaryEntry(this.key, this.value);
            }
        }
        object IDictionaryEnumerator.Value
        {
            get
            {
                if ((this.index == 0) || (this.index == (this._ReversibleSortedList.Count + 1)))
                {
                    throw new InvalidOperationException("Nie mona wykona operacji wyliczania.");
                }
                return this.value;
            }
        }
        void IEnumerator.Reset()
        {
            if (this.version != this._ReversibleSortedList.version)
            {
                throw new InvalidOperationException("Bd kontroli wersji typu wyliczeniowego");
            }
            this.index = 0;
            this.key = default(K);
            this.value = default(V);
        }
    }
    #endregion //  Enumerator <K, V>

    #region KeyList<K,V>
    [Serializable]
    private sealed class KeyList<K, V> : IList<K>, ICollection<K>, IEnumerable<K>, ICollection, IEnumerable
    {
        // Metody
        internal KeyList(ReversibleSortedList<K, V> dictionary)
        {
            this._dict = dictionary;
        }

        public void Add(K key)
        {
            throw new NotSupportedException("Operacja Add jest niedostpna");
        }

        public void Clear()
        {
            throw new NotSupportedException("Operacja Clear jest niedostpna");
        }

        public bool Contains(K key)
        {
            return this._dict.ContainsKey(key);
        }

        public void CopyTo(K[] array, int arrayIndex)
        {
            Array.Copy(this._dict.keys, 0, array, arrayIndex, this._dict.Count);
        }

        public IEnumerator<K> GetEnumerator()
        {
            return new ReversibleSortedList<K, V>.ReversibleSortedListKeyEnumerator(this._dict);
        }

        public int IndexOf(K key)
        {
        if (key.Equals(null))
        {
            throw new ArgumentNullException("key");
        }
        int num1 = Array.BinarySearch<K>(this._dict.keys, 0, this._dict.Count, key, this._dict._sortDirectionComparer);
        if (num1 >= 0)
        {
            return num1;
        }
        return -1;
    }

    public void Insert(int index, K value)
    {
        throw new NotSupportedException("Operacja Insert jest niedostpna");
    }

    public bool Remove(K key)
    {
        //throw new NotSupportedException("Operacja Remove jest niedostpna");
        return false;
    }

    public void RemoveAt(int index)
    {
        throw new NotSupportedException("Operacja RemoveAt jest niedostpna");
        }

        void ICollection.CopyTo(Array array, int arrayIndex)
        {
            if ((array != null) && (array.Rank != 1))
            {
                throw new ArgumentException("Tablice wielowymiarowe nie s obsugiwane");
            }
            try
            {
                Array.Copy(this._dict.keys, 0, array, arrayIndex, this._dict.Count);
            }
                catch (ArrayTypeMismatchException atme)
            {
                throw new ArgumentException("InvalidArrayType", atme);
            }
        }

        IEnumerator IEnumerable.GetEnumerator()
        {
            return new ReversibleSortedList<K, V>.ReversibleSortedListKeyEnumerator(this._dict);
        }

        // Waciwoci
        public int Count
        {
            get
            {
                return this._dict._size;
            }
        }

        public bool IsReadOnly
        {
            get
            {
                return true;
            }
        }

        public K this[int index]
        {
            get
            {
                return this._dict.GetKey(index);
            }
            set
            {
                throw new NotSupportedException("Operacja Set nie jest obsugiwana");
            }
        }

        bool ICollection.IsSynchronized
        {
            get
            {
                return false;
            }
        }

        object ICollection.SyncRoot
        {
            get
            {
                return this._dict;
            }
        }

        // Pola
        private ReversibleSortedList<K, V> _dict;
    }
    #endregion // KeyList<K,V>

    #region Definicja klasy ReversibleSortedListKeyEnumerator
    [Serializable]
    private sealed class ReversibleSortedListKeyEnumerator : IEnumerator<TKey>, IDisposable, IEnumerator
    {
        // Metody
        internal ReversibleSortedListKeyEnumerator(ReversibleSortedList<TKey, TValue> ReversibleSortedList)
        {
            this._ReversibleSortedList = ReversibleSortedList;
            this.version = ReversibleSortedList.version;
        }

        public void Dispose()
        {
            this.index = 0;
            this.currentKey = default(TKey);
        }

        public bool MoveNext()
        {
            if (this.version != this._ReversibleSortedList.version)
        {
            throw new InvalidOperationException("Bd kontroli wersji typu wyliczeniowego");
        }
        if (this.index < this._ReversibleSortedList.Count)
        {
            this.currentKey = this._ReversibleSortedList.keys[this.index];
            this.index++;
            return true;
        }
        this.index = this._ReversibleSortedList.Count + 1;
        this.currentKey = default(TKey);
        return false;
    }

    void IEnumerator.Reset()
    {
        if (this.version != this._ReversibleSortedList.version)
        {
            throw new InvalidOperationException("Bd kontroli wersji typu wyliczeniowego");
        }
        this.index = 0;
        this.currentKey = default(TKey);
    }

        // Waciwoci
        public TKey Current
        {
            get
            {
                return this.currentKey;
            }
        }

        object IEnumerator.Current
        {
            get
            {
                if ((this.index == 0) || (this.index == (this._ReversibleSortedList.Count + 1)))
            {
                throw new InvalidOperationException("Niedozwolona operacja wyliczania");
            }
            return this.currentKey;
        }
    }

        // Pola
        private ReversibleSortedList<TKey, TValue> _ReversibleSortedList;
        private TKey currentKey;
        private int index;
        private int version;
    }
    #endregion //definicja klasy  ReversibleSortedListKeyEnumerator 

    #region ReversibleSortedListValueEnumerator definition
    [Serializable]
    private sealed class ReversibleSortedListValueEnumerator : IEnumerator<TValue>, IDisposable, IEnumerator
    {
        // Metody
        internal ReversibleSortedListValueEnumerator(ReversibleSortedList<TKey, TValue> ReversibleSortedList)
        {
            this._ReversibleSortedList = ReversibleSortedList;
            this.version = ReversibleSortedList.version;
        }

        public void Dispose()
        {
            this.index = 0;
            this.currentValue = default(TValue);
        }

        public bool MoveNext()
        {
            if (this.version != this._ReversibleSortedList.version)
        {
            throw new InvalidOperationException("Bd kontroli wersji typu wyliczeniowego");
        }
        if (this.index < this._ReversibleSortedList.Count)
        {
            this.currentValue = this._ReversibleSortedList.values[this.index];
            this.index++;
            return true;
        }
        this.index = this._ReversibleSortedList.Count + 1;
        this.currentValue = default(TValue);
        return false;
    }

    void IEnumerator.Reset()
    {
        if (this.version != this._ReversibleSortedList.version)
        {
            throw new InvalidOperationException("Bd kontroli wersji typu wyliczeniowego");
        }
        this.index = 0;
        this.currentValue = default(TValue);
    }

    // Waciwoci
    public TValue Current
    {
        get
        {
            return this.currentValue;
        }
    }

    object IEnumerator.Current
    {
        get
        {
            if ((this.index == 0) || (this.index == (this._ReversibleSortedList.Count + 1)))
        {
            throw new InvalidOperationException("Niedozwolona operacja wyliczania");
        }
        return this.currentValue;
        }
    }


        // Pola
        private ReversibleSortedList<TKey, TValue> _ReversibleSortedList;
        private TValue currentValue;
        private int index;
        private int version;
    }
    #endregion //ReversibleSortedListValueEnumerator

    #region ValueList <K, V> definition
    [Serializable]
    private sealed class ValueList<K, V> : IList<V>, ICollection<V>, IEnumerable<V>, ICollection, IEnumerable
    {
        // Metody
        internal ValueList(ReversibleSortedList<K, V> dictionary)
        {
            this._dict = dictionary;
        }

        public void Add(V key)
        {
            throw new NotSupportedException("Operacja Add niedostpna");
        }

        public void Clear()
        {
            throw new NotSupportedException("Operacja Clear niedostpna");
        }

        public bool Contains(V value)
        {
            return this._dict.ContainsValue(value);
        }

        public void CopyTo(V[] array, int arrayIndex)
        {
            Array.Copy(this._dict.values, 0, array, arrayIndex, this._dict.Count);
        }

        public IEnumerator<V> GetEnumerator()
        {
            return new ReversibleSortedList<K, V>.ReversibleSortedListValueEnumerator(this._dict);
        }

        public int IndexOf(V value)
        {
            return Array.IndexOf<V>(this._dict.values, value, 0, this._dict.Count);
        }

        public void Insert(int index, V value)
        {
            throw new NotSupportedException("Operacja Insert niedostpna");
        }

        public bool Remove(V value)
        {
            //throw new NotSupportedException("Operacja Remove niedostpna");
        return false;
        }

        public void RemoveAt(int index)
        {
            throw new NotSupportedException("Operacja RemoveAt niedostpna");
        }

        void ICollection.CopyTo(Array array, int arrayIndex)
        {
            if ((array != null) && (array.Rank != 1))
            {
                throw new ArgumentException("Tablice wielowymiarowe nie s obsugiwane");
            }
            try
            {
                Array.Copy(this._dict.values, 0, array, arrayIndex, this._dict.Count);
            }
            catch (ArrayTypeMismatchException atme)
            {
                throw new ArgumentException("Nieprawidowy typ tablicy", atme);
            }
        }

        IEnumerator IEnumerable.GetEnumerator()
        {
            return new ReversibleSortedList<K, V>.ReversibleSortedListValueEnumerator(this._dict);
        }


        // Waciwoci
        public int Count
        {
            get
            {
                return this._dict._size;
            }
        }

        public bool IsReadOnly
        {
            get
            {
                return true;
            }
        }

        public V this[int index]
        {
            get
            {
                return this._dict.GetByIndex(index);
            }
            set
            {
                throw new NotSupportedException("Operacja set poprzez indeksator jest niedostpna");
            }
        }

        bool ICollection.IsSynchronized
        {
            get
            {
                return false;
            }
        }

        object ICollection.SyncRoot
        {
            get
            {
                return this._dict;
            }
        }


        // Pola
        private ReversibleSortedList<K, V> _dict;
    }
    #endregion // Definicja klasy ValueList <TKey, TValue>

    #endregion // Typy zagniedone
}
        public static void TestReversibleSortedList()
        {
            ReversibleSortedList<int, string> rsl = new ReversibleSortedList<int, string>();
            rsl.Add(2, "2");
            rsl.Add(5, "5");
            rsl.Add(3, "3");
            rsl.Add(1, "1");

            foreach (KeyValuePair<int, string> kvp in rsl)
            {
                Debug.WriteLine("\t" + kvp.Key + "\t" + kvp.Value);
            }

            // odwrcenie kierunku sortowania
            rsl.Comparer.SortDirection = ListSortDirection.Descending;
            // ponowne posortowanie listy
            rsl.Sort();

            foreach (KeyValuePair<int, string> kvp in rsl)
            {
                Debug.WriteLine("\t" + kvp.Key + "\t" + kvp.Value);
            }


            rsl.Add(4, "4");

            foreach (KeyValuePair<int, string> kvp in rsl)
            {
                Debug.WriteLine("\t" + kvp.Key + "\t" + kvp.Value);
            }

            // odwrcenie kierunku sortowania
            rsl.Comparer.SortDirection = ListSortDirection.Ascending;
            // ponowne posortowanie listy
            rsl.Sort();

            foreach (KeyValuePair<int, string> kvp in rsl)
            {
                Debug.WriteLine("\t" + kvp.Key + "\t" + kvp.Value);
            }
        }

#endregion // ReversibleSortedList



        #region "4.9 Tworzenie kolekcji tylko do odczytu z wykorzystaniem typw generycznych"
        public static void MakeCollectionReadOnly()
        {
            Lottery tryYourLuck = new Lottery();
            // Wywietlenie wynikw.
            for (int i = 0; i < tryYourLuck.Results.Count; i++)
            {
                Console.WriteLine("Zwyciskie numery " + i + " is " + tryYourLuck.Results[i]);
            }

            // Prba modyfikacji!
            //tryYourLuck.Results[0] = 29;

            //Wykonanie powyszego wiersza powoduje wywietlenie nastpujcego komunikatu o bdzie: // Error  26 //  Property or indexer
            // 'System.Collections.ObjectModel.ReadOnlyCollection<int>.this[int]'
            // cannot be assigned to -- it is read only
        }
        public class Lottery
        {
            // Utworzenie listy.
            List<int> _numbers = null;

            public Lottery()
            {
                // Utworzenie wewntrznej listy
                _numbers = new List<int>(5);
                // Dodanie wartoci
                _numbers.Add(17);
                _numbers.Add(21);
                _numbers.Add(32);
                _numbers.Add(44);
                _numbers.Add(58);
            }

            public ReadOnlyCollection<int> Results
            {
                // Zwrcenie opakowanej kopii wynikw.
                get { return new ReadOnlyCollection<int>(_numbers); }
            }
        }

		#endregion

        #region "4.10 Zastpienie typu Hashtable jego generycznym odpowiednikiem"
        public static void UseNonGenericHashtable()
		{
            // Utworzenie obiektu Hashtable i wypenienie go danymi.
            Hashtable numbers = new Hashtable();
            numbers.Add(1, "jeden");  // Powoduje wykonanie operacji pakowania dla indeksu.
            numbers.Add(2, "dwa");  // Powoduje wykonanie operacji pakowania dla indeksu.

            // Wywietlenie wszystkich par indeks-warto w tablicy Hashtable.
            // Powoduje wykonywanie operacji rozpakowania indeksu w kadej iteracji.
            foreach (DictionaryEntry de in numbers)
            {
                Console.WriteLine("Indeks: " + de.Key + "\tWarto: " + de.Value);
            }
            numbers.Clear();
        }

        public static void UseGenericDictionary()
        {
            // Utworzenie obiektu Dictionary i wypenienie go danymi.
            Dictionary<int, string> numbers = new Dictionary<int, string>();
            numbers.Add(1, "jeden");
            numbers.Add(2, "dwa");

            // Wywietlenie wszystkich par indeks-warto w obiekcie Dictionary.
            foreach (KeyValuePair<int, string> kvp in numbers)
            {
                Console.WriteLine("Indeks:  " + kvp.Key + "\tWarto:  " + kvp.Value);
            }
            numbers.Clear();
        }

		public static void CopyToGenericDictionary()
		{
			Console.WriteLine("\r\nCopyToGenericDictionary");

            //  Utworzenie obiektu Dictionary i wypenienie go danymi.
			Dictionary<int, string> numbers = new Dictionary<int, string>();
			numbers.Add(1, "jeden");
			numbers.Add(2, "dwa");

            //  Wywietlenie wszystkich par indeks-warto w obiekcie Dictionary.
			foreach (KeyValuePair<int, string> kvp in numbers)
			{
				Console.WriteLine("Indeks: " + kvp.Key + "\tWarto: " + kvp.Value);
			}

			// Utworzenie tablicy obiektw zawierajcej skopiowane informacje z obiektu Dictionary
			KeyValuePair<int, string>[] objs = new KeyValuePair<int, 
string>[numbers.Count];

            // Wywoanie metody CopyTo klasy Dictionary
            // Skopiowanie wszystkich obiektw KeyValuePair w obiekcie Dictionary do tablicy objs[]
			((IDictionary)numbers).CopyTo(objs, 0);

            //  Wywietlenie wszystkich par indeks-warto w tablicy objs[].
			foreach (KeyValuePair<int, string> kvp in objs)
			{
				Console.WriteLine("Indeks: " + kvp.Key + "\tWarto: " + kvp.Value);
			}
		}

		public static void CloneGenericDictionary()
		{
			Console.WriteLine("\r\nCloneGenericDictionary");

            //  Utworzenie obiektu Dictionary i wypenienie go danymi.
			Dictionary<int, string> numbers = new Dictionary<int, string>();
			numbers.Add(1, "jeden");
			numbers.Add(2, "dwa");

			// Wywietlenie wszystkich liczb cakowitych w oryginalnej zawartoci obiektu Dictionary
			foreach (KeyValuePair<int, string> kvp in numbers)
			{
				Console.WriteLine("Oryginalny indeks: " + kvp.Key + "\tWarto: " + kvp.Value);
			}

			// Sklonowanie obiektu Dictionary
			Dictionary<int, string> clonedNumbers = new Dictionary<int, 
string>(numbers);

			// Wywietlenie wszystkich liczb cakowitych w sklonowanym obiekcie Dictionary
			foreach (KeyValuePair<int, string> kvp in numbers)
			{
				Console.WriteLine("Sklonowany indeks: " + kvp.Key + "\tWarto: " + kvp.Value);
			}
		}
		#endregion

        #region "4.11 Korzystanie z ptli foreach dla generycznego typu Dictionary"
        public static void ShowForeachWithDictionary()
		{
            // Utworzenie obiektu Dictionary i wypenienie go danymi.
			Dictionary<int, string> myStringDict = new Dictionary<int, string>();
			myStringDict.Add(1, "Foo");
			myStringDict.Add(2, "Bar");
			myStringDict.Add(3, "Baz");

            // Wywietlenie wszystkich par indeks-warto w ptli foreach.
			foreach (KeyValuePair<int, string> kvp in myStringDict)
			{
				Console.WriteLine("Indeks : " + kvp.Key);
				Console.WriteLine("Warto: " + kvp.Value);
				Console.WriteLine("kvp    : " + kvp.ToString());
			}

			// Skorzystanie z obiektu DictionaryEntry powoduje bd kompilacji
			//foreach (DictionaryEntry de in myStringDict)
			//{
			//    Console.WriteLine("Indeks :  " + de.Key);
			//    Console.WriteLine("Warto:  " + de.Value);
			//    Console.WriteLine("kvp    :  " + de.ToString());
			//}
		}
		#endregion

        #region "4.12 Ograniczenia dla argumentw opisujcych typy"
        public static void TestConversionCls()
		{
			Console.WriteLine("\r\n\r\n");

			Conversion<long> c = new Conversion<long>();
			//Console.WriteLine("long.MinValue:  " + c.ShowAsInt(long.MinValue));
			Console.WriteLine("-100:  " + c.ShowAsInt(-100));
			Console.WriteLine("0:  " + c.ShowAsInt(0));
			Console.WriteLine("100:  " + c.ShowAsInt(100));
			//Console.WriteLine("long.MaxValue:  " + c.ShowAsInt(long.MaxValue));
		}

		public static void TestComparableListCls()
		{
			Console.WriteLine("\r\n\r\n");

			ComparableList<int> cp = new ComparableList<int>();
			cp.Add(100);
			cp.Add(10);

			Console.WriteLine("0 compare 1 == " + cp.Compare(0,1));
			Console.WriteLine("1 compare 0 == " + cp.Compare(1,0));
			Console.WriteLine("1 compare 1  == " + cp.Compare(1,1));
		}

		public static void TestDisposableListCls()
		{
			Console.WriteLine("\r\n\r\n");

			DisposableList<StreamReader> dl = new DisposableList<StreamReader>();

			// Utworzenie kilku obiektw testowych
			StreamReader tr1 = new StreamReader("c:\\boot.ini");
			StreamReader tr2 = new StreamReader("c:\\autoexec.bat");
			StreamReader tr3 = new StreamReader("c:\\config.sys");

			// Dodanie obiektw testowych do listy DisposableList
			dl.Add(tr1);
			dl.Insert(0, tr2);

			Console.WriteLine("dl.IndexOf(tr3) == " + dl.IndexOf(tr3));

			dl.Add(tr3);

			Console.WriteLine("dl.Contains(tr1) == " + dl.Contains(tr1));

			StreamReader[] srArray = new StreamReader[3];
			dl.CopyTo(srArray, 0);
			Console.WriteLine("srArray[1].ReadLine() == " + srArray[1].ReadLine());

			Console.WriteLine("dl.Count == " + dl.Count);

			foreach(StreamReader sr in dl)
			{
				Console.WriteLine("sr.ReadLine() == " + sr.ReadLine());
			}

			Console.WriteLine("dl.IndexOf(tr3) == " + dl.IndexOf(tr3));

			Console.WriteLine("dl.IsReadOnly == " + dl.IsReadOnly);

            // Wywoanie metody  Dispose przed usuniciem obiektu 
            // z listy DisposableList.

			dl.RemoveAt(0);
			Console.WriteLine("dl.Count == " + dl.Count);

			dl.Remove(tr1);
			Console.WriteLine("dl.Count == " + dl.Count);

			dl.Clear();
			Console.WriteLine("dl.Count == " + dl.Count);
		}


		public class Conversion<T>
			where T : struct, IConvertible
		{
			public int ShowAsInt(T val)
			{
				return (val.ToInt32(NumberFormatInfo.CurrentInfo));
			}
		}

		public class ComparableList<T> : List<T>
			where T : IComparable<T>
		{
			public int Compare(int index1, int index2)
			{
				return (index1.CompareTo(index2));
			}
		}

		public class DisposableList<T> : IList<T>
			where T : class, IDisposable
		{
			private List<T> _items = new List<T>();

			// Metoda prywatna, ktra usuwa wszystkie elementy z listy
			private void Delete(T item)
			{
				item.Dispose();
			}

			// Skadowe klasy IList<T>
			public int IndexOf(T item)
			{
				return (_items.IndexOf(item));
			}

			public void Insert(int index, T item)
			{
				_items.Insert(index, item);
			}

			public T this[int index]
			{
				get	{return (_items[index]);}
				set	{_items[index] = value;}
			}

			public void RemoveAt(int index)
			{
				Delete(this[index]);
				_items.RemoveAt(index);
			}

			// Skadowe klasy ICollection<T>
			public void Add(T item)
			{
				_items.Add(item);
			}

			public bool Contains(T item)
			{
				return (_items.Contains(item));
			}

			public void CopyTo(T[] array, int arrayIndex)
			{
				_items.CopyTo(array, arrayIndex);
			}

			public int Count
			{
				get	{return (_items.Count);}
			}

			public bool IsReadOnly
			{
				get	{return (false);}
			}

			// Skadowe klasy IEnumerable<T>
			public IEnumerator<T> GetEnumerator()
			{
				return (_items.GetEnumerator());
			}

			// Skadowe klasy IEnumerable
			IEnumerator IEnumerable.GetEnumerator()
			{
				return (_items.GetEnumerator());
			}

			// Pozostae skadowe
			public void Clear()
			{
				for (int index = 0; index < _items.Count; index++)
				{
					Delete(_items[index]);
				}

				_items.Clear();
			}

			public bool Remove(T item)
			{
				int index = _items.IndexOf(item);

				if (index >= 0)
				{
					Delete(_items[index]);
					_items.RemoveAt(index);

					return (true);
				}
				else
				{
					return (false);
				}
			}
		}
		#endregion

        #region "4.13 Inicjowanie zmiennych generycznych na ich wartoci domylne"
        public static void ShowSettingFieldsToDefaults()
		{
			Console.WriteLine("\r\n\r\n");

			DefaultValueExample<int> dv = new DefaultValueExample<int>();

            // Sprawdzenie, czy dane zainicjowano wartociami domylnymi; zwrcenie wartoci true.
			bool isDefault = dv.IsDefaultData();
            Console.WriteLine("Dane pocztkowe: " + isDefault);

            // Ustawienie wartoci danych.
			dv.SetData(100);

            // Ponowne sprawdzenie; zwrcenie wartoci  false.
			isDefault = dv.IsDefaultData();
            Console.WriteLine("Dane ustawione: " + isDefault);
		}

		public class DefaultValueExample<T>
		{
			T data = default(T);

			public bool IsDefaultData()
			{
				T temp = default(T);

				if (temp.Equals(data))
				{
					return (true);
				}
				else
				{
					return (false);
				}
			}

			public void SetData(T val)
			{
				data = val;
			}
		}
		#endregion

	}
}

